{ Contains the code for all file input and output. All I/O should be done using
  these procedures. There is one exception. In the note.pas file the drivers are
  written to the driver file directly from the memo component. It is done that
  way because the memo component has a function to write to a file and using
  that function is easier then converting the text in the memo to the correct
  form for the writedriverfile function. }
unit fileio;
 
interface

uses sysutils, classes, Dialogs, stypes, frontend, options;

type  TAction = (flRead, flWrite);

// Model I/O
function  ReadParamFile(var filename:string; const npar:integer;
           var parray:paramarray; const nstat:integer;
           var sarray:statearray; var Tresid:double):Boolean;
procedure WriteParamFile(filename:string; const npar:integer;
          var parray:paramarray; const nstat:integer; var sarray:statearray;
          var Tresid:double);
procedure OpenDriverFile(filename:string; const numdrive:integer;
          var darray:drivearray; Action:TAction);
procedure GetCurrentDrivers(CurrTime:double; var CurrDrivers:DriveArray);
function  ReadDriverFile(var drivetime:double; const numdrive:integer;
          var darray:drivearray):Boolean;
function  WriteDriverFile(var drivetime:double; const numdrive:integer;
          var darray:drivearray):Boolean;
procedure CloseDriverFile;
procedure OpenOutputFile(filename:string;const numstat:integer;
          var sarray:statearray; const numproc:integer; var parray:processarray;
          Action:TAction);
function  ReadOutputFile(var time:double; const numdrive:integer; var darray:
          drivearray; const numstat:integer; var sarray:statearray; const
          numproc:integer; var parray:processarray):Boolean;
function WriteOutputFile(time:double; const numstat:integer;
          var sarray:statearray; const numproc:integer;
          var parray:processarray):Boolean;
procedure CloseOutputFile;

// Batch I/O
procedure OpenBatchFile(filename:string);
function  ReadBatchFile(var paramname, drivername, outputname:string; var begintime,
             endtime:double; var tstat:statearray; var opt:TRunOptions):Boolean;
procedure CloseBatchFile;
procedure OpenLogFile(filename, batchfile:string);
function  WriteLogFile(filename:string; outputname:string;
                        StatusMessage:string):Boolean;
procedure CloseLogFile;

// Sensitivity I/O
procedure OpenListFile(filename:string);
function  ReadListFile(var paramfilename, driverfilename, paramname:string;
                        var newparamvalue:double):Boolean;
procedure CloseListFile;
procedure OpenMeasFile(filename:string);
function  ReadMeasFile(var thisline:string): Boolean;
procedure CloseMeasFile;
procedure OpenSensOutFile(filename:string);
function  WriteSensOutFile(var paramfilename, paramname:string;
             var newparamvalue:double; TestStatus:string; ResidCL, ResidCW,
             ResidCR, ResidNL, ResidNW, ResidNR, ResidCT, ResidNT:
             double):Boolean;
procedure CloseSensOutFile;

// Calibration I/O
function ReadCalSetFile(filename: string; var nset: integer; var cset: calsetarray): Boolean;
procedure WriteCalSetFile(filename: string; nset: integer; ilist: Tstringlist; var cset: calsetarray);
procedure OpenCalOutFile(filename:string);
function  WriteCalOutFile(const ctime:double;  const treatnum:integer;
   const numsets:integer; const CalSet:calsetarray):Boolean;
procedure CloseCalOutFile;
procedure OpenOutputFile2(filename:string; const numdrive:integer;
          var darray:drivearray; const numstat:integer; var sarray:statearray;
          const numproc:integer; var parray:processarray; Action:TAction; appendfile:Boolean);
function  ReadOutputFile2(var time:double; const numdrive:integer; var darray:
          drivearray; const numstat:integer; var sarray:statearray; const
          numproc:integer; var parray:processarray):Boolean;
function WriteOutputFile2(time:double; const numdrive:integer;
          var darray:drivearray; const numstat:integer;
          var sarray:statearray; const numproc:integer;
          var parray:processarray):Boolean;
procedure CloseOutputFile2;

// Modelshell I/O
procedure  ReadModelDef(filename:string; var tempmodeldef:Tmodeldef;
     var tstat:statearray; var tpar:paramarray; var tproc:processarray;
     var tdrive:drivearray);

// General procedures
procedure ChangeExtension(var filename:string; NewExt:string);
procedure RemoveSpaces(var somestring:string);
function ParCount(processnum:integer) : integer;

var
 last_time,next_time:double;
 last_drive,next_drive: drivearray;
 driverfile:textfile;
 outfile, outfile2:textfile;
 batchfile:textfile;
 logfile:textfile;
 listfile:textfile;
 sensOutfile:textfile;
 calOutfile:textfile;
 measfile:textfile;
 ParamFileVersion:string;

implementation

uses calculate;

// Parameter File I/O

{ The parameter file is a space delimited ASCII file containing the parameter
  values and the initial values of the state variables. Each line of the file
  contains one parameter or state variable. The first item on the line is the
  variable name (a string up to stringlength characters), the second item is the numerical
  value of the item and the final item on the line is the units of the parameter
  or state variable (also a string up to stringlength characters long). }

{ This function reads the entire parameter file if it exists or creates a new
   empty file. }
function ReadParamFile(var filename:string; const npar:integer;
           var parray:paramarray; const nstat:integer;
           var sarray:statearray; var Tresid:double):Boolean;
var
  i,charnum:integer;
  dum, dum2, dum3, dum4, tempvalue:string[stringlength+1];
  tempstring1, tempstring2 :string;
  paramfile:textfile;
begin
  ReadParamFile := False;
  assignfile(paramfile,filename);
  NeedToSavePar := False;
  reset(paramfile);
  readln(paramfile, ParamFileVersion);
  SetLength(tempstring1, stringlength + 2);
  strlcopy(PChar(tempstring1), PChar(ParamFileVersion), stringlength + 1);
 { Delete any spaces in the model name read from the parameter file and any
   spaces in a copy of the modelname field of modeldef. This is to prevent
   the invalid parameter file version error due to differences in spacing. }
  tempstring1 := trim(tempstring1);
  charnum := pos(' ',tempstring1);
  while charnum <> 0 do
    begin
      delete(tempstring1,charnum,1);
      charnum := pos(' ',tempstring1);
    end;
 // And now delete the spaces from the modelname.
  tempstring2 := ModelDef.modelname;
  charnum := pos(' ',tempstring2);
  while charnum <> 0 do
    begin
      delete(tempstring2,charnum,1);
      charnum := pos(' ',tempstring2);
    end;

 // Compare the 2 strings containing the modelname.
  if StrIComp(PChar(tempstring1),PChar(tempstring2)) <> 0 then
    begin
      reset(paramfile);
      ParamFileVersion := 'Unknown'; // Model names don't match
    end
  else  // Model names do match, now extract the version number and compare
    begin
      delete(ParamFileVersion,1,stringlength + 1);
      ParamFileVersion := trim(ParamFileVersion);
      charnum := pos(' ',ParamFileVersion);
      if charnum <> 0 then
        delete(ParamFileVersion,charnum,Length(ParamFileVersion)-charnum+1);
    end;

 try
  if ParamFileVersion <> ModelDef.versionnumber then
    begin
      raise EInvalidParFile.create('The parameter file selected is not compatible with this model. '
                   + 'No parameters read.');
    end
  else
{ Model name and version match the parameter file so read in the state variables
  and parameters. }
    begin
      for i:= 1 to nstat do  // Read state variables ignoring names and units
       begin
         readln(paramfile,dum3,tempvalue,dum4,dum,dum2);
         sarray[i].value := strtofloat(tempvalue);
         dum := trim(dum);
         dum2 := trim(dum2);
         if (dum[1] = 'T') or (dum[1] = 't') then
           sarray[i].holdconstant := True
         else
           sarray[i].holdconstant := False;
         if (dum2[1] = 'T') or (dum2[1] = 't') then
           sarray[i].reset := True
         else
           sarray[i].reset := False;
       end;
      for i:= 1 to npar do   // Read parameters ignoring names and units
         readln(paramfile,dum,parray[i].value,dum2);
      readln(paramfile,dum,Tresid);
      if Tresid = 0 then Tresid := 999;
      ReadParamFile := true;
    end;
 finally
  closefile(paramfile);
 end;
end;  // End readparamfile

{ Writes the entire parameter file, overwriting any existing file. Frontend
  does the checking to be sure you want to overwrite an existing file. See
  DlgSaveParam. }
procedure WriteParamFile(filename:string; const npar:integer;
           var parray:paramarray; const nstat:integer;
           var sarray:statearray; var Tresid:double);
var
  i:integer;
  paramfile:textfile;
  tstring1, tstring2: string;
begin
  assignfile(paramfile,filename);
  rewrite(paramfile);
  // Write out the model name and version.
  writeln(paramfile, format('%-25.25s',[ModelDef.modelname]), ' ',
                     format('%-25.25s',[ModelDef.VersionNumber]), ' ',
                     format('%-77s',['Units                     Holdconstant              Reset']));
  // Write state variable name, value, units, holdconstant and reset
  for i:= 1 to nstat do
   begin
    if sarray[i].holdconstant then
     tstring1 := 'True'
    else
     tstring1 := 'False';
    if sarray[i].reset then
     tstring2 := 'True'
    else
     tstring2 := 'False';
    writeln(paramfile,format('%-25.25s',[sarray[i].name]),' ',
                      format('%-25.25e',[sarray[i].value]),' ',
                             format('%-25.25s',[sarray[i].units]), ' ',
                          //   sarray[i].holdconstant:-25, ' ', sarray[i].reset:-25);
                             format('%-25.25s',[tstring1{sarray[i].holdconstant}]), ' ',
                             format('%-25.25s',[tstring2{sarray[i].reset}]));
   end;
  // Write parameter name, value and units
  for i:= 1 to npar do
    writeln(paramfile,format('%-25.25s',[parray[i].name]),' ',
                      format('%-25e',[parray[i].value]),' ',
                             format('%-25.25s',[parray[i].units]));
  writeln(paramfile,format('%-25.25s',['Total Residual ']),' ',format('%-25g',[Tresid]));
  closefile(paramfile);
  NeedToSavePar := false;
end;  // end of writeparamfile


// Driver File I/O

{ The driver file is a space delimited ASCII file containing the values for the
  driver variables at specified times. Each line of the file contains a value for
  all the drivers for the specified time. The first item on the line is the time,
  followed by the drivers. The first line of the file contains the names of the
  variables, the second contains the units of the items, and the third line
  contains the actual values of the drivers. Subsequent lines contain the drivers
  for later times. Driver variables are held at the last value read until the
  time specified on the next line. Negative times cause the drivers to be
  ramped from the values specified on the line before the negative time to the
  values specified on the line with the negative time. The function
  GetCurrentDrivers calculates any ramping if necessary. Access to the
  drivers while running should be done using GetCurrentDrivers.

  The driver file must be opened and closed using OpenDriverFile and
  CloseDriverFile.  Be sure to call CloseDriverFile when done. }

// Opens the driver file for reading or writing depending on the value of Action
procedure OpenDriverFile(filename:string; const numdrive:integer;
          var darray:drivearray; Action:TAction);
var
 i:integer;
 tempstring:string;
begin
 assignfile(driverfile,filename);
 if Action = flRead then  // Open driver file for reading
  begin
   if not FileExists(filename) then
    begin
     MessageDlg('Cannot read driver file. File does not exist.',
               mtError,[mbOK],0);
    end
   else   // File exists so read in the header lines
    begin      // Open driver, necessary for the note window to edit the drivers
     reset(driverfile);  // also done in getcurrentdrivers to allow run x repeatedly
     readln(driverfile,tempstring);    // Read driver names and throw away
     readln(driverfile,tempstring);    // Read driver units and throw away
    end;
  end
 else
{ Open driver file for writing and write out driver names and units. }
  begin
    rewrite(driverfile);  // Create a new driver file
    write(driverfile,format('%-25.25s',['Time']),' ');  // Write time header
    for i := 1 to numdrive do        // Write driver names
     write(driverfile,format('%-25.25s',[darray[i].name]),' ');
    writeln(driverfile);   // Write return
    write(driverfile,format('%-25.25s',[ModelDef.timeunit]),' '); // Write time unit
    for i := 1 to numdrive do     // Write driver units
     write(driverfile,format('%-25.25s',[darray[i].units]),' ');
    writeln(driverfile);   // Write return
  end;
end;   // end of opendriverfile

{ This procedure gets the values for the drivers at the time specified in
  CurrTime. If the next_time read in from the driver file is negative, this
  procedure calculates a linear ramp between the values in last_drive and
  next_drive. }
procedure GetCurrentDrivers(CurrTime:double; var CurrDrivers:DriveArray);
var
 tfrac : double;
 i:integer;
 tempstring:string;
begin
// No previous values have been read. Initialize everything.
 if (last_time = 0) and (next_time = 0) then
  begin
   reset(driverfile); //Necessary for run x repeatedly.
   readln(driverfile,tempstring);    // Read driver names and throw away
   readln(driverfile,tempstring);    // Read driver units and throw away
   ReadDriverFile(next_time,ModelDef.numdrive,next_drive);
   last_time := next_time;
   last_drive := next_drive;
   CurrDrivers := next_drive;
  end;

{ If CurrTime (passed to this procedure from the calling procedure) is greater
  than the value in next_time, read in new values for the drivers. }
 while CurrTime >= abs(next_time) do
// while (CurrTime-1e-6 >= abs(next_time)) or (CurrTime+1e-6 >= abs(next_time)) do
  begin
   last_time := next_time;
   CurrDrivers := next_drive;
   last_drive := next_drive;
   if not ReadDriverFile(next_time,ModelDef.numdrive,next_drive)
       then next_time := 2*time_stop;
  end;

{ If next_time is less than zero calculate the drivers using a linear ramp
  between last_time and next_time. }
 if (next_time < 0) and  (abs(next_time)<>abs(last_time)) then
  begin
   tfrac := (CurrTime-abs(last_time))/(abs(next_time)-abs(last_time));
   for i := 1 to ModelDef.numdrive do
     CurrDrivers[i].value := last_drive[i].value + tfrac*(next_drive[i].value -
                                       last_drive[i].value);
  end;
end;    // getcurrentdrivers

// Reads one line of the driver file
function ReadDriverFile(var drivetime:double; const numdrive:integer;
                            var darray:drivearray):Boolean;
var
  j:integer;
  temptime:double;
  tempdrive:drivearray;
  EmptyLine:Boolean;
begin
// Copy the names and units from darray to tempdrive so they aren't overwritten.
   tempdrive := darray;
   if not eof(driverfile) then // Check for end of file
    begin
     read(driverfile,temptime);   // Read drive time
     for j:=1 to numdrive do
      read(driverfile,tempdrive[j].value);  // Read drivers
     readln(driverfile);      // Advance to next line
     EmptyLine := False;
// If all the drivers read in are zero then the line is an empty line.
     if (temptime = 0) then
       begin
        EmptyLine := True;
        for j := 1 to numdrive do
          EmptyLine := EmptyLine and (tempdrive[j].value = 0);
       end;
// Read successful assign the values read into darray and drivetime
     if not EmptyLine then
      begin
        drivetime := temptime;
        darray := tempdrive;
        ReadDriverFile := True;  // Read successful
      end
     else
// Read in an empty line, advance to the end of the file.
      begin
        while not eof(driverfile) do begin
          readln(driverfile);
        end;
        ReadDriverFile := False;
      end;
    end
   else
    ReadDriverFile := False;  // End of file, read unsuccessful
end;     // readdriverfile

// Writes one line to the driver file.
function WriteDriverFile(var drivetime:double; const numdrive:integer;
           var darray:drivearray):Boolean;
var
 i:integer;
begin
   write(driverfile,format('%-25.13e',[drivetime]),' ');   // Write drive time
   for i := 1 to numdrive do
    write(driverfile,format('%-25.13e',[darray[i].value]),' '); // Write drivers
   writeln(driverfile);   // Write return
   WriteDriverFile := True;
end;   // writedriverfile

// Closes the driver file
procedure CloseDriverFile ;
begin
 closefile(driverfile);
end;


// Output File I/O

{ The output file is a space delimited ASCII file containing values for the
  state variables and processes at each time step of the model run. The first
  item on a line is the time followed by the state variables and then the
  process variables. The first two lines of the file are header lines. The first
  line contains the names of the variables and the second line contains the
  units.

  The output file must be opened and closed using OpenOutputFile and
  CloseOutputFile. Be sure to call CloseOutputFile when you are done.}

// Opens the output file for reading or writing depending on the value of Action
procedure OpenOutputFile(filename:string; const numstat:integer;
          var sarray:statearray; const numproc:integer; var parray:processarray;
          Action:TAction);
var
  j:integer;
  temp:string;
begin
 assignfile(outfile,filename);
 if Action = flread then  // Open output file for reading
   begin
     reset(outfile);
     readln(outfile,temp); // Read variable names and throw them away
     readln(outfile,temp); // Read variable units and throw them away
   end
 else  // Open output file to write
   begin
    if FmOptions.RunOptions.AppendOutputFile then
     begin
      append(outfile);
     end
    else
     begin
      rewrite(outfile);    // Create a new output file
      write(outfile,format('%25.25s',['Time']),' ');  // Write Variable Names
      for j := 1 to ModelDef.numdrive do
        write(outfile,format('%25.25s',[drive[j].name]),' ');
      for j := 1 to numstat do
        write(outfile,format('%25.25s',[sarray[j].name]),' ');
      for j := ModelDef.numstate + 1 to numproc do
        write(outfile,format('%25.25s',[parray[j].name]),' ');
      writeln(outfile);
      write(outfile,format('%25.25s',[ModelDef.timeunit]),' '); // Write Variable
      for j := 1 to ModelDef.numdrive do                              //   Units
        write(outfile,format('%25.25s',[drive[j].units]),' ');
      for j := 1 to numstat do
        write(outfile,format('%25.25s',[sarray[j].units]),' ');
      for j := ModelDef.numstate + 1 to numproc do
        write(outfile,format('%25.25s',[parray[j].units]),' ');
      writeln(outfile);
     end;
   end;
end;  // openoutputfile

// Reads one line of the output file
function ReadOutputFile(var time:double; const numdrive:integer; var darray:
         drivearray; const numstat:integer; var sarray:statearray; const
         numproc:integer;var parray:processarray):Boolean;
var
 j:integer;
begin
   if eof(outfile) then
     ReadOutputFile := False
   else
    begin
     read(outfile, time);    // Read time
     for j:= 1 to ModelDef.numdrive
      do read(outfile,darray[j].value);
     for j := 1 to numstat
        do read(outfile,sarray[j].value); // State variables
     for j := ModelDef.numstate + 1 to numproc
        do read(outfile,parray[j].value); // Process variables
     readln(outfile);  // Advance to next line
     ReadOutputFile := True;
    end;
end;  // readoutputfile

// Writes one line to the output file
function WriteOutputFile(time:double; const numstat:integer;
         var sarray:statearray; const numproc:integer;
         var parray:processarray):Boolean;
var
  j:integer;
begin
   write(outfile,format('%25g',[time]),' ');  // Write time
   for j := 1 to ModelDef.numdrive
      do write(outfile,format('%25.13e',[drive[j].value]),' ');
   for j := 1 to numstat     // State variables
      do write(outfile,format('%25.13e',[sarray[j].value]),' ');
   for j := ModelDef.numstate + 1 to numproc     // Process variables
      do write(outfile,format('%25.13e',[parray[j].value]),' ');
   writeln(outfile);  // Write return
   WriteOutputFile := True;
end;  // writeoutputfile

// Closes the output file
procedure CloseOutputFile;
begin
   closefile(outfile);
end;

// Batch File I/O

{ The batch file is a comma delimited ASCII file containing all the information
  necessary to do a model run. Each line represents one run of the model and
  contains the parameter file, driver file, and output file names, the start
  time, the stop time and the time step. The first is an example.

  The batch file must be opened and closed using OpenBatchFile and
  CloseBatchFile. Be sure to call CloseBatchFile when you are done.}
  
{ Opens the batch file, reads in the first line and discards it. }
procedure OpenBatchFile(filename:string);
var
tempstring:string;
begin
 assignfile(batchfile,filename);
 reset(batchfile);
 readln(batchfile,tempstring);    // Read names and throw away
end;

{ Reads a line from the batch file. Returns true if the read was successful and
  returns false otherwise. }
function  ReadBatchFile(var paramname, drivername, outputname:string; var begintime,
             endtime:double; var tstat:statearray; var opt:TRunOptions):Boolean;
var
 tempstring1, tempstring2: string;
 numbegin, numend, index:integer;
begin
 if not eof(batchfile) then // Check for end of file
  begin
   readln(batchfile,tempstring1);   // Read line
   numbegin := 1;
   numend := pos(',',tempstring1);
   paramname := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(paramname,pos(',',paramname),1);
   paramname := trim(paramname);

   numend := pos(',',tempstring1);
   drivername := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(drivername,pos(',',drivername),1);
   drivername := trim(drivername);

   numend := pos(',',tempstring1);
   outputname := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(outputname,pos(',',outputname),1);
   outputname := trim(outputname);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   begintime := strtofloat(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   endtime := strtofloat(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   opt.Time_step := strtofloat(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   opt.Discretestep := strtofloat(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   tempstring2 := trim(tempstring2);
   if (tempstring2[1] = 't') or (tempstring2[1] = 'T') then
     Opt.NormalRun := True
   else
     Opt.NormalRun := False;

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   tempstring2 := trim(tempstring2);
   if (tempstring2[1] = 't') or (tempstring2[1] = 'T') then
     Opt.RepeatDrivers := True
   else
     Opt.RepeatDrivers := False;

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   Opt.RepeatDriveTime := strtoint(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   tempstring2 := trim(tempstring2);
   if (tempstring2[1] = 't') or (tempstring2[1] = 'T') then
     Opt.ResetStates := True
   else
     Opt.ResetStates := False;

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   Opt.ResetStateTime := strtoint(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   tempstring2 := trim(tempstring2);
   if (tempstring2[1] = 't') or (tempstring2[1] = 'T') then
     Opt.RuntoSS := True
   else
     Opt.RuntoSS := False;

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   Opt.SSCriteria := strtofloat(tempstring2)/100;

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   Opt.SSTime := strtoint(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   tempstring2 := trim(tempstring2);
   if (tempstring2[1] = 't') or (tempstring2[1] = 'T') then
     Opt.HoldStatesConstant := True
   else
     Opt.HoldStatesConstant := False;

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   Opt.OutputStep := strtofloat(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   Opt.OutputOffset := strtofloat(tempstring2);

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   tempstring2 := trim(tempstring2);
   if (tempstring2[1] = 't') or (tempstring2[1] = 'T') then
     Opt.OutputEORonly := True
   else
     Opt.OutputEORonly := False;

   numend := pos(',',tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   tempstring2 := trim(tempstring2);
   if (tempstring2[1] = 't') or (tempstring2[1] = 'T') then
     Opt.AppendOutputFile := True
   else
     Opt.AppendOutputFile := False;

   Opt.ErrorMult := strtoint(tempstring1);

   ReadBatchFile := True;
  end
 else
   ReadBatchFile := False;
end;

{ Closes the batch file. }
procedure CloseBatchFile;
begin
 closefile(batchfile);
end;

// Log File I/O

{ The log file is an ASCII file summarizing the batch job defined in the
  batch file. It contains the date and time, the batch file
  name, and for each run: the output file name and a message with the status
  of the run. }
  
{ Opens the Batch log file and writes out initial information. }
procedure OpenLogFile(filename, batchfile:string);
begin
 assignfile(logfile,filename);
 rewrite(logfile);
 writeln(logfile, 'Output log for Modelshell Batch Utility Version ' + ModelDef.versionnumber);
 writeln(logfile, 'Generated: ' + DateTimeToStr(Now));
 writeln(logfile, 'Output from batch file: ' + batchfile);
end;

{ Write a line to the log file containing the output file name and a status
  message. }
function  WriteLogFile(filename:string; outputname:string;
                        StatusMessage:string):Boolean;
begin
 writeln(logfile, outputname + ': ' + StatusMessage);
 WriteLogFile := True;
end;

procedure CloseLogFile;
begin
 closefile(logfile);
end;

// List File I/O

{ The list file is a comma delimited ASCII file containing information
  necessary for running a sensitivity analysis. Each line represents one
  sensitivity test and contains the parameter file name, the driver file name
  the parameter to vary, the new value for the parameter. The first line is
  an example.

  The list file must be opened and closed using OpenListFile and
  CloseListFile. Be sure to call CloseListFile when you are done.}
  
{ Opens the list file, reads in the first line and discards it. }
procedure OpenListFile(filename:string);
var
tempstring:string;
begin
 assignfile(listfile,filename);
 reset(listfile);
 readln(listfile,tempstring);    // Read names and throw away
end;

{ Reads a line from the list file. Returns true if the read was successful and
  returns false otherwise. }
function  ReadListFile(var paramfilename, driverfilename, paramname:string;
                        var newparamvalue:double):Boolean;
var
 tempstring1,tempstring2: string;
 numbegin,numend: integer;
begin
 if not eof(listfile) then // Check for end of file
  begin
   readln(listfile, tempstring1);   // Read line
   removespaces(tempstring1);
   numbegin := 1;
   numend := pos(',',tempstring1);
   paramfilename := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(paramfilename,pos(',',paramfilename),1);

   numend := pos(',',tempstring1);
   driverfilename := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(driverfilename,pos(',',driverfilename),1);

   numend := pos(',',tempstring1);
   paramname := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(paramname,pos(',',paramname),1);

   numend :=  length(tempstring1);
   tempstring2 := copy(tempstring1,numbegin,numend-numbegin+1);
   delete(tempstring1,numbegin,numend);
   delete(tempstring2,pos(',',tempstring2),1);
   newparamvalue := strtofloat(tempstring2);

   ReadListFile := True;
  end
 else
   ReadListFile := False;
end;

{ Closes the list file. }
procedure CloseListFile;
begin
 closefile(listfile);
end;

// Measured data file I/O

{ The measured data file is a comma delimited ASCII file containing the
  measured data to be compared with the model. Each line represents one
  comparison and contains the treatment code, the simulation year,
  the measured leaf C, wood C, root C, leaf N, wood N, and root N. Note that
  the treatment code is used to identify the output file to read the data from.
  Therefore, the treatment code must appear in the output filename and it must
  appear in only one output filename. The year is the simulation year that
  corresponds to the measured data on this line.

  The measured data file must be opened and closed using OpenMeasFile
  and CloseMeasFile. Be sure to call CloseMeasFile when you are done.}

{ This procedure opens the Measured data file for reading, reads in the first
  line containing the headings and discards it.}
procedure OpenMeasFile(filename:string);
var
 tempstring: string;
begin
 assignfile(measfile,filename);
 reset(measfile);
 readln(measfile,tempstring);    // Read names and throw away
end;

{ Reads one line of the measured data file. Parsing of the line read in is done
  in a different procedure to allow easy changing of the measured data. }
function  ReadMeasFile(var thisline:string): Boolean;
begin
 if not eof(measfile) then
  begin
    readln(measfile,thisline);
    ReadMeasFile := true;
  end
 else
    ReadMeasFile := false;
end;

{ Closes the measured data file. }
procedure CloseMeasFile;
begin
 CloseFile(measfile);
end;

// Sensitivity Output File I/O

{ The sensitivity output file is a comma delimited ASCII file containing the
  results of the sensitivity analysis. Each line represents one sensitivity
  test and contains the parameter file name, the parameter to vary, the new
  value for the parameter and residuals for leaf, wood and root carbon and
  nitrogen.

  The sensitivity output file must be opened and closed using OpenSensOutFile
  and CloseSensOutFile. Be sure to call CloseSensOutFile when you are done.}

{ Opens the sensitivity output file and writes out column headings. }
procedure OpenSensOutFile(filename:string);
var
tempstring:string;
begin
 tempstring := 'Parameter File, Parameter, New Value, Test Status, ResidCL,' +
     ' ResidCW, ResidCR, ResidNL, ResidNW, ResidNR, ResidCT, ResidNT';
 assignfile(sensoutfile,filename);
 rewrite(sensoutfile);
 writeln(sensoutfile,tempstring);
end;

{ Write a line to the sens output file. }
function  WriteSensOutFile(var paramfilename, paramname:string;
             var newparamvalue:double; TestStatus:string; ResidCL, ResidCW,
             ResidCR, ResidNL, ResidNW, ResidNR, ResidCT, ResidNT:
             double):Boolean;
var
 tempstring: string;
begin
 tempstring := paramfilename + ',' + paramname + ',' + floattostr(newparamvalue)
    + ',' + TestStatus + ',' + floattostr(ResidCL) + ',' + floattostr(ResidCW) +
     ',' + floattostr(ResidCR) + ',' + floattostr(ResidNL) + ',' +
    floattostr(ResidNW) + ',' + floattostr(ResidNR) + ',' + floattostr(ResidCT)
    + ',' + floattostr(ResidNT);
 writeln(sensoutfile,tempstring);
 WriteSensOutFile := True;
end;

{ Closes the Sensitivity output file. }
procedure CloseSensOutFile;
begin
 closefile(SensOutFile);
end;

procedure ChangeExtension(var filename:string; NewExt:string);
var
 num:integer;
begin
 num := pos('.',filename);
 delete(filename,num,length(filename)-num+1);
 filename := filename + '.' + NewExt;
end;

procedure RemoveSpaces(var somestring:string);
var
 num: integer;
begin
 num := Pos(' ', somestring);
 while num > 0 do
   begin
    delete(somestring,num,1);
    num := Pos(' ', somestring);
   end;
end;

{ Reads in the model definition file to create a new model. }
procedure  ReadModelDef(filename:string; var tempmodeldef:Tmodeldef;
     var tstat:statearray; var tpar:paramarray; var tproc:processarray;
     var tdrive:drivearray);
var
 modelfile:textfile;
 i,j,npar:integer;
 tempstring, tname, tunits, tsymbol, tnpar:string;

procedure SetString(var tfile:textfile; var tstring:string);
var
 num:integer;
begin
 readln(tfile,tstring);
 num := pos('=',tstring);
 delete(tstring,1,num);
end;

procedure SetNUS(var tfile:textfile; var tstring, tname, tunits,
                                            tsymbol:string);
var
 num:integer;
begin
 readln(tfile,tstring);
 num := pos(',',tstring);
 tname := copy(tstring,1,num-1);
 delete(tstring,1,num);
 num := pos(',',tstring);
 tunits := copy(tstring,1,num-1);
 delete(tstring,1,num);
 num := pos(',',tstring);
 if num = 0 then
  begin
   num := length(tstring);
   tsymbol := copy(tstring,1,num);
  end
 else
  begin
   tsymbol := copy(tstring,1,num-1);
   delete(tstring,1,num);
  end;
end;

procedure SetPD(var tstring, tnpar:string);
var
 num:integer;
begin
 num := pos(',',tstring);
 tnpar := copy(tstring,1,num-1);
 delete(tstring,1,num);
 num := length(tstring);
end;

begin
try try
 assignfile(modelfile,filename);
 reset(modelfile);
 readln(modelfile,tempstring);  // Throw away the first line of the file
// ModelDef
 SetString(modelfile,tempstring);
 tempmodeldef.modelname := tempstring;
 SetString(modelfile,tempstring);
 tempmodeldef.versionnumber := tempstring;
 SetString(modelfile,tempstring);
 tempmodeldef.timeunit := tempstring;
 readln(modelfile, tempstring);    // Throw away blank line

// States
 SetString(modelfile,tempstring);
 tempmodeldef.numstate := strtoint(tempstring);
 readln(modelfile,tempstring);  // Throw away comment line
 for i := 1 to tempmodeldef.numstate do
  begin
   SetNUS(modelfile, tempstring, tname, tunits, tsymbol);
   tstat[i].name := tname;
   tstat[i].units := tunits;
   tstat[i].symbol := tsymbol;
  end;
 readln(modelfile, tempstring);    // Throw away blank line

// Processes
 SetString(modelfile,tempstring);
 tempmodeldef.numprocess := tempModelDef.numstate + strtoint(tempstring);
 readln(modelfile,tempstring);  // Throw away comment line
 for i := tempmodeldef.numstate + 1 to tempmodeldef.numprocess do
  begin
   SetNUS(modelfile, tempstring, tname, tunits, tsymbol);
   tproc[i].name := tname;
   tproc[i].units := tunits;
   tproc[i].symbol := tsymbol;
   SetPD(tempstring, tnpar);
   tproc[i].parameters := strtoint(tnpar);
   if tproc[i].parameters <> 0 then
   begin
    npar := ParCount(i);
    for j := 1 to tproc[i].parameters do
     begin
      SetNUS(modelfile, tempstring, tname, tunits, tsymbol);
      tpar[npar + j].name := tname;
      tpar[npar + j].units := tunits;
      tpar[npar + j].symbol := tsymbol;
     end;
   end;
  end;
 readln(modelfile, tempstring);    // Throw away blank line

// Drivers
 SetString(modelfile,tempstring);
 tempmodeldef.numdrive := strtoint(tempstring);
 readln(modelfile,tempstring);  // Throw away comment line
 for i := 1 to tempmodeldef.numstate do
  begin
   SetNUS(modelfile, tempstring, tname, tunits, tsymbol);
   tdrive[i].name := tname;
   tdrive[i].units := tunits;
   tdrive[i].symbol := tsymbol;
  end;

 { Set the names, units, and symbols of the processes and the parameters. The
  maximum length for the name, units and symbol is stringlength (currently 25)
  characters.

  The first numstate processes are the derivatives of the state variables. Set
  the names and units accordingly.}
 for i:= 1 to tempmodeldef.numstate do tproc[i].name:='d'+tstat[i].name+'dt';
 for i:= 1 to tempmodeldef.numstate do tproc[i].units := tstat[i].units + 't-1';
 for i:= 1 to tempmodeldef.numstate do tproc[i].symbol := 'd' + tstat[i].symbol + 'dt';

{ Sum up the total number of parameters in the model and store it in the
  tempmodeldef structure. }
 tempmodeldef.numparam := 0;
 for i := 1 to tempmodeldef.NumProcess do
  tempmodeldef.numparam := tempmodeldef.numparam + tproc[i].parameters;

finally
 CloseFile(modelfile);
end;
except
 MessageDlg('Error reading Model Definition file. Application Terminated',
             mtError,[mbOK],0);
 raise;
end;
end;

// Calibration Output File I/O

{ The CalSet file is a comma delimited ASCII file containing the calibration
  set (i.e. the treatment, state variable, parameter, alpha and beta). Each line
  represents one calibration set.

  The calibration output file is a comma delimited ASCII file containing the
  results of the calibration procedure. Each line represents one time interval
  and contains the time of comparison, and the new parameter value.

  The calibration output file must be opened and closed using OpenCalOutFile
  and CloseCalOutFile. Be sure to call CloseCalOutFile when you are done.}

{ Reads in the calibration set file. }
function ReadCalSetFile(filename: string; var nset: integer; var cset: calsetarray): Boolean;
var
 num:integer;
 tempstring, smallstring: string;
 CalSetFile: TextFile;
begin
 assignfile(CalSetFile, filename);
 reset(CalSetFile);
 nset := 0;
 while not eof(CalSetFile) do
  begin
    readln(CalSetFile,tempstring);
    nset := nset + 1;

    num := pos(',', tempstring);
    smallstring := copy(tempstring,1,num-1);
    Cset[nset].Treatment := strtoint(smallstring);
    delete(tempstring,1,num);

    num := pos(',', tempstring);
    smallstring := copy(tempstring,1,num-1);
    Cset[nset].State := trim(smallstring);
    delete(tempstring,1,num);

    num := pos(',', tempstring);
    smallstring := copy(tempstring,1,num-1);
    Cset[nset].Parameter := trim(smallstring);
    delete(tempstring,1,num);

    num := pos(',', tempstring);
    smallstring := copy(tempstring,1,num-1);
    Cset[nset].Alpha := strtofloat(smallstring);
    delete(tempstring,1,num);

    Cset[nset].Beta := strtofloat(tempstring);
    delete(tempstring,1,num);
  end;
 ReadCalSetFile := True;
 Closefile(CalSetFile);
end;

{ Writes the calibration sets to a file. }
procedure WriteCalSetFile(filename: string; nset: integer; ilist: Tstringlist; var cset: calsetarray);
var
 i, snum:integer;
 tempstring: string;
 CalSetFile: TextFile;
begin
 assignfile(CalSetFile, filename);
 try
  rewrite(CalSetFile);
  for i := 0 to nset - 1 do
   begin
     snum := strtoint(ilist[i]);
     tempstring := inttostr(cset[snum].treatment) + ', ' + cset[snum].State + ', ' +
        cset[snum].Parameter + ', ' + floattostr(cset[snum].alpha) + ', ' +
        floattostr(cset[snum].beta);
     Writeln(CalSetFile, tempstring);
   end;
 finally
  Closefile(CalSetFile);
 end;
end;

{ Opens the calibration output file and writes out column headings. }
procedure OpenCalOutFile(filename:string);
var
 tempstring: string;
begin
 tempstring := 'Treatment, Time, Parameter, New Value, Parameter, New Value......';
 assignfile(CalOutFile,filename);
 rewrite(CalOutFile);
 writeln(CalOutFile,tempstring);
end;

{ Writes a line to the calibration output file. }
function  WriteCalOutFile(const ctime:double; const treatnum:integer;
   const numsets:integer; const CalSet:calsetarray):Boolean;
var
 tempstring: string;
 i: integer;
begin
 tempstring := inttostr(treatnum) + ', ' + floattostr(ctime) + ', ';
 for i := 1 to numsets + 1 do
  begin
    if Calset[i].Treatment = treatnum then
     tempstring := tempstring + par[CalSet[i].ParamIndex].name + ', ' +
        floattostr(par[CalSet[i].ParamIndex].value) + ', ';
  end;
 writeln(CalOutFile,tempstring);
 WriteCalOutFile := True;
end;

{ Closes the calibration output file. }
procedure CloseCalOutFile;
begin
 closefile(CalOutFile);
end;

// Output File I/O

{ The output file is a space delimited ASCII file containing values for the
  state variables and processes at each time step of the model run. The first
  item on a line is the time followed by the state variables and then the
  process variables. The first two lines of the file are header lines. The first
  line contains the names of the variables and the second line contains the
  units.

  The output file must be opened and closed using OpenOutputFile and
  CloseOutputFile. Be sure to call CloseOutputFile when you are done.}

// Opens the output file for reading or writing depending on the value of Action
procedure OpenOutputFile2(filename:string; const numdrive:integer;
   var darray:drivearray; const numstat:integer; var sarray:statearray; const
   numproc:integer; var parray:processarray; Action:TAction; appendfile:Boolean);
var
  j:integer;
  temp:string;
begin
 assignfile(outfile2,filename);
 if Action = flread then  // Open output file for reading
   begin
     reset(outfile2);
     readln(outfile2,temp); // Read variable names and throw them away
     readln(outfile2,temp); // Read variable units and throw them away
   end
 else  // Open output file to write
   begin
    if appendfile then
     begin
      append(outfile2);
     end
    else
     begin
      rewrite(outfile2);    // Create a new output file
      write(outfile2,format('%25.25s',['Time']),' ');  // Write Variable Names
      for j := 1 to numdrive do
        write(outfile2,format('%25.25s',[darray[j].name]),' ');
      for j := 1 to numstat do
        write(outfile2,format('%25.25s',[sarray[j].name]),' ');
      for j := numstat + 1 to numproc do
        write(outfile2,format('%25.25s',[parray[j].name]),' ');
      writeln(outfile2);
      write(outfile2,format('%25.25s',[ModelDef.timeunit]),' '); // Write Variable
      for j := 1 to numdrive do                              //   Units
        write(outfile2,format('%25.25s',[darray[j].units]),' ');
      for j := 1 to numstat do
        write(outfile2,format('%25.25s',[sarray[j].units]),' ');
      for j := numstat + 1 to numproc do
        write(outfile2,format('%25.25s',[parray[j].units]),' ');
      writeln(outfile2);
     end;
   end;
end;  // openoutputfile

// Reads one line of the output file
function ReadOutputFile2(var time:double; const numdrive:integer; var darray:
         drivearray; const numstat:integer; var sarray:statearray; const
         numproc:integer;var parray:processarray):Boolean;
var
 j:integer;
begin
   if eof(outfile2) then
     ReadOutputFile2 := False
   else
    begin
     read(outfile2, time);    // Read time
     for j:= 1 to numdrive
      do read(outfile2,darray[j].value);
     for j := 1 to numstat
        do read(outfile2,sarray[j].value); // State variables
     for j := numstat + 1 to numproc
        do read(outfile2,parray[j].value); // Process variables
     readln(outfile2);  // Advance to next line
     ReadOutputFile2 := True;
    end;
end;  // readoutputfile

// Writes one line to the output file
function WriteOutputFile2(time:double; const numdrive:integer;
          var darray:drivearray; const numstat:integer; var sarray:statearray;
          const numproc:integer; var parray:processarray):Boolean;
var
  j:integer;
begin
   write(outfile2,format('%25g',[time]),' ');  // Write time
   for j := 1 to numdrive
      do write(outfile2,format('%25.13e',[darray[j].value]),' ');
   for j := 1 to numstat     // State variables
      do write(outfile2,format('%25.13e',[sarray[j].value]),' ');
   for j := ModelDef.numstate + 1 to numproc     // Process variables
      do write(outfile2,format('%25.13e',[parray[j].value]),' ');
   writeln(outfile2);  // Write return
   WriteOutputFile2 := True;
end;  // writeoutputfile

// Closes the output file
procedure CloseOutputFile2;
begin
 try
   closefile(outfile2);
 except
  if (IOResult = 103) or (IOResult = 105) then ; // No then clause, just want to "handle" the exception if the file isn't open.
 end;
end;

{This function counts the parameters in all processes less than processnum.}
function ParCount(processnum:integer) : integer;
var
 NumberofParams, counter : integer;
begin
  NumberofParams := 0;
  for counter := ModelDef.numstate + 1 to processnum - 1 do
         NumberofParams := NumberofParams + proc[counter].parameters;
  ParCount := NumberofParams;
end; // end of parcount function

end.
